-- Create a test user.
GRANT CONNECT, RESOURCE, ALTER SESSION, XDBADMIN TO test IDENTIFIED BY test;

-- Connect as that test user.
CONNECT test/test;

-- Globally register the purchase order schema with XML DB.
BEGIN
  DBMS_XMLSCHEMA.registerschema(
    schemaurl=>'http://www.example.com/po.xsd',
    schemadoc=>'
<xsd:schema
  xmlns:xsd="http://www.w3.org/2001/XMLSchema"
  xmlns:po="http://www.example.com/po"
  targetNamespace="http://www.example.com/po"
  elementFormDefault="qualified"
  attributeFormDefault="qualified">

  <xsd:annotation>
    <xsd:documentation xml:lang="en">
     Purchase order schema for Example.com.
     Copyright 2000 Example.com. All rights reserved.
    </xsd:documentation>
  </xsd:annotation>

  <xsd:element name="purchaseOrder" type="po:PurchaseOrderType"/>

  <xsd:element name="comment" type="xsd:string"/>

  <xsd:complexType name="PurchaseOrderType">
    <xsd:sequence>
      <xsd:element name="shipTo" type="po:USAddress"/>
      <xsd:element name="billTo" type="po:USAddress"/>
      <xsd:element ref="po:comment" minOccurs="0"/>
      <xsd:element name="items"  type="po:Items"/>
    </xsd:sequence>
    <xsd:attribute name="orderDate" type="xsd:date"/>
  </xsd:complexType>

  <xsd:complexType name="USAddress">
    <xsd:sequence>
      <xsd:element name="name"   type="xsd:string"/>
      <xsd:element name="street" type="xsd:string"/>
      <xsd:element name="city"   type="xsd:string"/>
      <xsd:element name="state"  type="xsd:string"/>
      <xsd:element name="zip"    type="xsd:decimal"/>
    </xsd:sequence>
    <xsd:attribute name="country" type="xsd:NMTOKEN"
                   fixed="US"/>
  </xsd:complexType>

  <xsd:complexType name="Items">
    <xsd:sequence>
      <xsd:element name="item" minOccurs="0" maxOccurs="unbounded">
        <xsd:complexType>
          <xsd:sequence>
            <xsd:element name="productName" type="xsd:string"/>
            <xsd:element name="quantity">
              <xsd:simpleType>
                <xsd:restriction base="xsd:positiveInteger">
                  <xsd:maxExclusive value="100"/>
                </xsd:restriction>
              </xsd:simpleType>
            </xsd:element>
            <xsd:element name="USPrice"  type="xsd:decimal"/>
            <xsd:element ref="po:comment"   minOccurs="0"/>
            <xsd:element name="shipDate" type="xsd:date" minOccurs="0"/>
          </xsd:sequence>
          <xsd:attribute name="partNum" type="po:SKU" use="required"/>
        </xsd:complexType>
      </xsd:element>
    </xsd:sequence>
  </xsd:complexType>

  <!-- Stock Keeping Unit, a code for identifying products -->
  <xsd:simpleType name="SKU">
    <xsd:restriction base="xsd:string">
      <xsd:pattern value="\d{3}-[A-Z]{2}"/>
    </xsd:restriction>
  </xsd:simpleType>
  
</xsd:schema>
  ',
  local=>false,
  enablehierarchy=>DBMS_XMLSCHEMA.ENABLE_HIERARCHY_CONTENTS);
  
  END;
  /

-- Create a resource in the XML DB repository conforming to the purchase
-- order schema.
DECLARE
  ret BOOLEAN;
BEGIN
  ret := DBMS_XDB.createresource('/public/po.xml', '
<purchaseOrder
  xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
  xmlns="http://www.example.com/po"
  xmlns:po="http://www.example.com/po"
  xsi:schemaLocation="http://www.example.com/po http://www.example.com/po.xsd"
  po:orderDate="1999-10-20">
  
  <shipTo>
    <name>Alice Smith</name>
    <street>123 Maple Street</street>
    <city>Mill Valley</city>
    <state>CA</state>
    <zip>90952</zip>
  </shipTo>
  
  <billTo po:country="US">
    <name>Robert Smith</name>
    <street>8 Oak Avenue</street>
    <city>Old Town</city>
    <state>PA</state>
    <zip>95819</zip>
  </billTo>
  
  <comment>Hurry, my lawn is going wild!</comment>
  
  <items>
    <item po:partNum="872-AA">
       <productName>Lawnmower</productName>
       <quantity>1</quantity>
       <USPrice>148.95</USPrice>
       <comment>Confirm this is electric</comment>
    </item>
    <item po:partNum="926-AA">
       <productName>Baby Monitor</productName>
       <quantity>1</quantity>
       <USPrice>39.98</USPrice>
       <shipDate>1999-05-21</shipDate>
    </item>
  </items>
</purchaseOrder>');
END;
/

COMMIT;
EXIT;
